# 機能設計書 7-変数管理（Resource Variable Operations）

## 概要

本ドキュメントは、TensorFlowのコアフレームワークにおける変数管理（Resource Variable Operations）機能の設計を記述する。リソース変数の作成・読み取り・更新・削除を管理する。

### 本機能の処理概要

変数管理機能は、TensorFlowにおけるリソースベースの変数（ResourceVariable）のライフサイクルを管理する操作群を提供する。VarHandleOpによる変数ハンドルの作成、ReadVariableOpによる値の読み取り、AssignVariableOp/AssignAddVariableOp/AssignSubVariableOpによる値の更新、DestroyResourceOpによる変数の削除など、モデルパラメータの管理に不可欠な操作を提供する。

**業務上の目的・背景**：機械学習モデルの学習では、重みやバイアスなどの学習可能パラメータを変数として管理し、勾配降下法により繰り返し更新する必要がある。本機能はこれらの変数をリソースハンドルとして管理し、デバイス間での共有やスレッドセーフな更新を可能にする。

**機能の利用シーン**：(1) モデルの重み変数の作成と初期化、(2) 学習ループ内での重み更新（AssignVariableOp）、(3) 変数値の読み取り（推論時）、(4) チェックポイントからの変数復元、(5) 分散学習でのクロスデバイス変数共有。

**主要な処理内容**：
1. VarHandleOp: 変数ハンドルの作成（リソースハンドル生成）
2. ReadVariableOp: 変数値の読み取り
3. ReadVariablesOp: 複数変数の一括読み取り
4. AssignVariableOp: 変数値の代入
5. AssignAddVariableOp: 変数への加算代入
6. AssignSubVariableOp: 変数からの減算代入
7. ResourceGather: 変数からのインデックス指定取得
8. ResourceScatterUpdate/Add/Sub/Mul/Div/Min/Max: 変数へのスキャッター更新
9. DestroyResourceOp: 変数リソースの破棄
10. VarIsInitializedOp: 変数の初期化状態確認

**関連システム・外部連携**：分散学習ストラテジー（tf.distribute）と連携し、変数のミラーリングやシャーディングを行う。チェックポイントシステムと連携し、変数の保存・復元を行う。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコアAPIのため直接的な画面関連はなし |

## 機能種別

CRUD操作（変数のCreate/Read/Update/Delete）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| resource | resource handle | Yes | 変数リソースハンドル | 有効なリソースハンドル |
| dtype | tf.DType | Yes | 変数のデータ型 | サポートされるデータ型 |
| shape | tf.TensorShape | Yes | 変数の形状 | 有効な形状 |
| value | tf.Tensor | 更新時 | 代入値 | dtypeと一致すること |
| container | string | No | リソースコンテナ名 | - |
| shared_name | string | No | 共有名 | - |
| debug_name | string | No | デバッグ用名前 | - |

### 入力データソース

Python API（tf.Variable）経由、またはオプティマイザの重み更新処理から呼び出される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| resource | resource handle | VarHandleOp: 変数リソースハンドル |
| value | tf.Tensor | ReadVariableOp: 変数の現在値 |
| is_initialized | tf.Tensor(bool) | VarIsInitializedOp: 初期化状態 |

### 出力先

呼び出し元のPythonコード（tf.Variable, オプティマイザ等）に返却される。

## 処理フロー

### 処理シーケンス

```
1. tf.Variable()呼び出し
   └─ ResourceVariable.__init__()
2. VarHandleOp実行
   └─ リソースハンドル生成（container/shared_name管理）
3. AssignVariableOp実行（初期値代入）
   └─ 初期値テンソルを変数に代入
4. 学習ループ内での操作
   └─ ReadVariableOp → 勾配計算 → AssignAddVariableOp/AssignSubVariableOp
5. 変数の破棄
   └─ DestroyResourceOp（スコープ終了時）
```

### フローチャート

```mermaid
flowchart TD
    A[tf.Variable作成] --> B[VarHandleOp:ハンドル生成]
    B --> C[AssignVariableOp:初期値代入]
    C --> D{操作種別}
    D -->|読み取り| E[ReadVariableOp]
    D -->|更新| F[AssignVariableOp/Add/Sub]
    D -->|スキャッター| G[ResourceScatterUpdate]
    D -->|破棄| H[DestroyResourceOp]
    E --> I[値テンソル返却]
    F --> I
    G --> I
    H --> J[リソース解放]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 型一致制約 | ReadVariableOp/AssignVariableOpのdtypeは変数のdtypeと一致 | 常時 |
| BR-02 | ロック制御 | use_locking=trueの場合、スレッドセーフな更新を保証 | 更新操作時 |
| BR-03 | 形状アノテーション | _shape属性による形状推論のヒント（関数内推論用） | 関数トレーシング時 |
| BR-04 | バリアント型サポート | DT_VARIANT型変数はサブシェイプ情報を持つ | variant型変数時 |

### 計算ロジック

- ReadVariableOp: 変数メモリから値テンソルをコピー（またはゼロコピー参照）
- AssignVariableOp: value テンソルの値を変数メモリに書き込み
- AssignAddVariableOp: variable = variable + value
- ResourceScatterUpdate: variable[indices] = updates

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作は発生しない（インメモリ変数管理） |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgumentError | 型不一致 | ReadVariableOpのdtypeが変数と不一致 | 正しいdtypeを指定する |
| NotFoundError | リソースなし | 未初期化変数へのアクセス | 変数を初期化してからアクセスする |
| FailedPreconditionError | 未初期化 | VarIsInitializedOpがFalse | 変数を初期化する |

### リトライ仕様

該当なし。

## トランザクション仕様

use_locking=trueの場合、変数更新はミューテックスによりアトミックに実行される。複数の変数更新を跨ぐトランザクションはサポートされない。

## パフォーマンス要件

- ReadVariableOpは可能な限りゼロコピーで実行すること
- GPU上の変数アクセスはデバイスメモリ上で直接実行すること

## セキュリティ考慮事項

- shared_nameによる変数共有はプロセス内でのみ有効
- allowed_devices属性によるデバイス制約

## 備考

- ResourceVariableはレガシーのVariableV2/Assign系opを置き換える新しい変数管理方式
- _shape属性はユーザが関数のキャプチャリストを通じて形状情報を注入する仕組み

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

リソース変数のハンドルとメタデータ（型・形状・コンテナ）を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | resource_variable_ops.cc | `tensorflow/core/ops/resource_variable_ops.cc` | VarHandleOp のAttr定義 |

**読解のコツ**: リソース変数はresource型のハンドルを通じてアクセスされる。ハンドルには変数の型（dtype）と形状（shape）のメタデータが関連付けられ、ShapeAndType構造体で管理される。

#### Step 2: エントリーポイントを理解する

C++のop定義から変数操作の仕様を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | resource_variable_ops.cc | `tensorflow/core/ops/resource_variable_ops.cc` | 全op定義 |

**主要処理フロー**:
- **34-59行目**: `ReadVariableShapeFn` - 変数読み取りの形状推論（_shape属性対応）
- **61-89行目**: `ReadVariablesShapeFn` - 複数変数一括読み取りの形状推論
- **93-100行目**: `VarHandleOp` op登録 - container, shared_name, debug_name, dtype, shape, allowed_devices属性

#### Step 3: Python側のResourceVariable実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | resource_variable_ops.py | `tensorflow/python/ops/resource_variable_ops.py` | ResourceVariableクラス |

### プログラム呼び出し階層図

```
tf.Variable(initial_value, ...)
    |
    +-- ResourceVariable.__init__()
            |
            +-- VarHandleOp
            |       +-- リソースハンドル生成
            |       +-- container/shared_name管理
            |
            +-- AssignVariableOp(handle, initial_value)
                    +-- 初期値をハンドルに代入

variable.assign(new_value)
    |
    +-- AssignVariableOp(handle, new_value)

variable.read_value()
    |
    +-- ReadVariableOp(handle)
            +-- ReadVariableShapeFn (形状推論)
```

### データフロー図

```
[入力]                    [処理]                        [出力]

initial_value      -->  VarHandleOp               -->  resource handle
(テンソル)               (ハンドル生成)
                            |
                    AssignVariableOp
                    (初期値代入)
                            |
                    ReadVariableOp              -->  value tensor
                    (値読み取り)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| resource_variable_ops.cc | `tensorflow/core/ops/resource_variable_ops.cc` | ソース | C++ op定義・形状推論 |
| resource_variable_ops.py | `tensorflow/python/ops/resource_variable_ops.py` | ソース | ResourceVariableクラス |
| gen_resource_variable_ops.py | `tensorflow/python/ops/gen_resource_variable_ops.py` | 自動生成 | Python バインディング |
| resource_mgr.h | `tensorflow/core/framework/resource_mgr.h` | ヘッダ | リソース管理基盤 |
| node_def_util.h | `tensorflow/core/framework/node_def_util.h` | ヘッダ | ノード定義ユーティリティ |
